<?php
/**
 * Contact List
 *
 * @package     AutomatorWP\Integrations\ActiveMember360\Triggers\Contact_List
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_ActiveMember360_Contact_List extends AutomatorWP_Integration_Trigger {

    public $integration = 'activemember360';
    public $trigger = 'activemember360_contact_list';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is subscribed to a list', 'automatorwp-pro' ),
            'select_option'     => __( 'User is subscribed to a <strong>list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: List name. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User is subscribed to %1$s %2$s time(s)', 'automatorwp-pro' ), '{list}', '{times}' ),
            /* translators: %1$s: List name. */
            'log_label'         => sprintf( __( 'User is subscribed to %1$s', 'automatorwp-pro' ), '{list}' ),
            'action'            => 'init',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'list' => array(
                    'from' => 'list',
                    'fields' => array(
                        'list' => array(
                            'name' => __( 'List:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'lists_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Get lists options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function lists_options_cb() {

        $options = array(
            'any' => __( 'any list', 'automatorwp-pro' ),
        );

        // Get site lists
        $lists = apply_filters( 'mbr/site_lists/get', NULL );

        if( ! empty( $lists ) ) {
            foreach( $lists as $list_id => $list_name ) {
                $options[$list_id] = $list_name;
            }
        }

        return $options;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     */
    public function listener() {

        // Bail if in admin area
        if( is_admin() ) {
            return;
        }

        $user_id = get_current_user_id();

        // Bail if user is not logged in
        if( $user_id === 0 ) {
            return;
        }

        $remote_logged_in = apply_filters( 'mbr/contact_id', NULL );

        // Bail if user account is not remote logged in
        if ( empty( $remote_logged_in ) ) {
            return;
        }

        // Trigger the user is subscribed to a list
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Shorthand
        $list = $trigger_options['list'];
        $user_lists = apply_filters( 'mbr/contact/lists', NULL );

        // Don't deserve if user isn't on any list
        if( empty( $user_lists ) ) {
            return false;
        }

        // Don't deserve if list doesn't match with the trigger option
        if( $list !== 'any' && ! array_key_exists( $list, $user_lists ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_ActiveMember360_Contact_List();